/*****************************************************************************/
/* File         dynarray.h                                                   */
/*                                                                           */
/* Purpose      Header for dynamically resizable arrays                      */
/*****************************************************************************/
#ifndef DYNARRAY_H
#define DYNARRAY_H

#ifndef SRCHDEF_H
#include "srchdef.h"
#endif


#if defined(__cplusplus)
extern "C"
{
#endif

struct sdynarray_struct
{
  void_ptr_t  data;
  size_t      length;
};
typedef struct sdynarray_struct sdynarray_t;

struct dynarray_struct
{
  sdynarray_t array;
  size_t      length;
  size_t      element_size;
};
typedef struct dynarray_struct dynarray_t;

struct dacursor_struct
{
  dynarray_t  *da_ptr;
  size_t       offset;
};
typedef struct dacursor_struct dacursor_t;

/*****************************************************************************/
/* The following macros are for internal use only                            */
/*****************************************************************************/
#define SDYNARRAY_ADDITION(ptr,offset) ((void_ptr_t)(((byte_t *)ptr)+offset))
#define SDYNARRAY_ADDRESS(da_ptr,offset) (SDYNARRAY_ADDITION((da_ptr)->data,offset))
#define SDYNARRAY_UNPROTECTED_WRITE(da_ptr,data,offset,length)\
  (memcpy(SDYNARRAY_ADDRESS((da_ptr),(offset)),(data),(length)))
#define SDYNARRAY_UNPROTECTED_READ(da_ptr,data,offset,length)\
  (memcpy((data),SDYNARRAY_ADDRESS((da_ptr),(offset)),(length)))
#define SDYNARRAY_LENGTH(da_ptr) ((da_ptr)->length)

/*****************************************************************************/
/* These are the external SDYNARRAY macros                                   */
/*****************************************************************************/
#define SDYNARRAY_QUICK_WRITE(da_ptr,data,offset,type)\
  (*((type *)SDYNARRAY_ADDRESS(da_ptr,offset))=(type)(data))
#define SDYNARRAY_VALID_OFFSET(da_ptr,offset)\
  ((offset)<(da_ptr)->length)
#define SDYNARRAY_QUICK_READ(da_ptr,offset,type)\
  (*((type *)SDYNARRAY_ADDRESS(da_ptr,offset)))
#define SDYNARRAY_WRITE(da_ptr,data,offset,length)\
  (SDYNARRAY_VALID_OFFSET(da_ptr,offset+length-1)?\
    (SDYNARRAY_UNPROTECTED_WRITE(da_ptr,data,offset,length),TRUE):\
    (sdynarray_write(da_ptr,data,offset,length)))
#define SDYNARRAY_READ(da_ptr,data,offset,length)\
  (SDYNARRAY_VALID_OFFSET(da_ptr,offset+length-1)?\
    (SDYNARRAY_UNPROTECTED_READ(da_ptr,data,offset,length),TRUE):\
    FALSE)

/*****************************************************************************/
/* These macros are not for use externally.                                  */
/*****************************************************************************/
#define DYNARRAY_VALID_INDEX(da_ptr,offset) ((bool_t)((offset)<(da_ptr)->length))
#define DYNARRAY_ADDRESS(da_ptr,offset)\
  SDYNARRAY_ADDRESS(&((da_ptr)->array),(da_ptr)->element_size*(offset))
#define DYNARRAY_LENGTH(da_ptr) ((da_ptr)->length)
#define DYNARRAY_SET_LENGTH(da_ptr,len) ((da_ptr)->length=(len))
#define DYNARRAY_EMPTY(da_ptr) ((bool_t)((da_ptr)->length==0))
#define DYNARRAY_UNPROTECTED_WRITE(da_ptr,data,offset)\
  SDYNARRAY_UNPROTECTED_WRITE(&(da_ptr)->array,(data),(offset)*(da_ptr)->element_size,(da_ptr)->element_size)
#define DYNARRAY_UNPROTECTED_READ(da_ptr,data,offset)\
  SDYNARRAY_UNPROTECTED_READ(&(da_ptr)->array,(data),(offset)*(da_ptr)->element_size,(da_ptr)->element_size)
#define DYNARRAY_QUICK_WRITE(da_ptr,data,offset,type)\
  (*((type *)DYNARRAY_ADDRESS(da_ptr,offset))=(type)(data))
#define DYNARRAY_QUICK_READ(da_ptr,offset,type)\
  (*((type *)DYNARRAY_ADDRESS(da_ptr,offset)))


#define DACURSOR_ADDRESS(dc_ptr) (DYNARRAY_ADDRESS((dc_ptr)->da_ptr,(dc_ptr)->offset))

/*****************************************************************************/
/* External interface macros                                                 */
/*****************************************************************************/
#define DACURSOR_INIT(dc_ptr,dap) (((dc_ptr)->da_ptr=(dap)),((dc_ptr)->offset=0),(bool_t)((dap)->length!=0))
#define DACURSOR_INIT_OFFSET(dc_ptr, dap, off) (dacursor_init_offset(dc_ptr, dap, off))
#define DACURSOR_NEXT(dc_ptr) DACURSOR_NEXT_COUNT((dc_ptr),1)
#define DACURSOR_PREV(dc_ptr) DACURSOR_PREV_COUNT((dc_ptr),1)
#define DACURSOR_NEXT_COUNT(dc_ptr,count) ((unsigned)(dc_ptr)->offset+(count)<DACURSOR_LENGTH(dc_ptr)?((dc_ptr)->offset+=(count),TRUE):FALSE)
#define DACURSOR_PREV_COUNT(dc_ptr,count) ((unsigned)(dc_ptr)->offset>=(count)?((dc_ptr)->offset-=(count),TRUE):FALSE)
#define DACURSOR_SET_OFFSET(dc_ptr,off) ((bool_t)((DYNARRAY_VALID_INDEX((dc_ptr)->da_ptr,(off))) && (((dc_ptr)->offset=(off)),TRUE)))
#define DACURSOR_READ_OFFSET(dc_ptr) ((dc_ptr)->offset)
#define DACURSOR_READ(dc_ptr,data)\
  ((DYNARRAY_UNPROTECTED_READ((dc_ptr)->da_ptr,data,DACURSOR_READ_OFFSET(dc_ptr))))
#define DACURSOR_WRITE(dc_ptr,data)\
  ((DYNARRAY_UNPROTECTED_WRITE((dc_ptr)->da_ptr,data,DACURSOR_READ_OFFSET(dc_ptr))))
#define DACURSOR_QUICK_READ(dc_ptr,type)\
  (DYNARRAY_QUICK_READ((dc_ptr)->da_ptr,(dc_ptr)->offset,type))
#define DACURSOR_QUICK_WRITE(dc_ptr,data,type)\
  (DYNARRAY_QUICK_WRITE((dc_ptr)->da_ptr,data,(dc_ptr)->offset,type))
#define DACURSOR_VALID_INDEX(dc_ptr) ((dc_ptr)->offset<(size_t)(dc_ptr)->da_ptr->length)
#define DACURSOR_INSERT_BEFORE(dc_ptr,data) (dacursor_insert_before(dc_ptr,data))
#define DACURSOR_INSERT_AFTER(dc_ptr,data) (dacursor_insert_after(dc_ptr,data))
#define DACURSOR_DELETE(dc_ptr) (dacursor_delete(dc_ptr))
#define DACURSOR_GOTO_END(dc_ptr) (((dc_ptr)->offset=DACURSOR_LENGTH(dc_ptr)-1),DACURSOR_EMPTY(dc_ptr))
#define DACURSOR_GOTO_BEGIN(dc_ptr) (((dc_ptr)->offset=0),DACURSOR_EMPTY(dc_ptr))
#define DACURSOR_LENGTH(dc_ptr) (DYNARRAY_LENGTH((dc_ptr)->da_ptr))
#define DACURSOR_EMPTY(dc_ptr) (DACURSOR_LENGTH((dc_ptr))!=0)
#define DACURSOR_APPEND(dc_ptr,data) (dynarray_append((dc_ptr)->da_ptr,data))
#define DACURSOR_AT_BEGIN(dc_ptr) ((bool_t)(((dc_ptr)->offset)==0))
#define DACURSOR_AT_END(dc_ptr) ((bool_t)(((dc_ptr)->offset)==DACURSOR_LENGTH(dc_ptr)-1))
#define DACURSOR_COMPARE(dc1,dc2,func) (((comp_fn_t)(func))(DACURSOR_ADDRESS(dc1),DACURSOR_ADDRESS(dc2)))
#define DACURSOR_COMPARE_CONTENTS(dc1,p,func) (((comp_fn_t)(func))(DACURSOR_ADDRESS(dc1),p))
#define DACURSOR_SWAP(dc1,dc2) (dacursor_swap((dc1),(dc2)))
#define DACURSOR_COPY(dc1,dc2) (dacursor_copy((dc1),(dc2)))
#define DACURSOR_ASSIGN(dc1,dc2) (*(dc1)= *(dc2))
#define DACURSOR_LOCATE(found,dc,bp,func)\
do\
{\
  found=(bool_t)(DACURSOR_COMPARE_CONTENTS(dc,bp,func)==0);\
  if (!found && DACURSOR_NEXT(dc)==FALSE)\
    break;\
} while(!found);

#define DACURSOR_SORTED_LOCATE(found,dc,bp,func)\
{\
  int compare_result;\
  {\
    compare_result=DACURSOR_COMPARE_CONTENTS(dc,bp,func)==0);\
    if (compare_result==0)\
    {\
      found=TRUE;\
      break;\
    }\
    found=FALSE;\
    if (compare_result<0)\
      break;\
    if (!found && DACURSOR_NEXT(dc)==FALSE)\
      break;\
  } while(!found);\
}

#if defined(SHORT_EXTERN_NAMES)
#define sdynarray_create sdacreat
#define sdynarray_destroy sdadest
#define sdynarray_write sdawrit
#define sdynarray_read sdaread
#define dynarray_create dnacreat
#define dynarray_reset dnareset
#define dynarray_destroy dnadest
#define dynarray_write dnawrit
#define dynarray_append dnaapp
#define dynarray_read dnaread
#define dacursor_delete dacdel
#define dacursor_insert_before dacinb
#define dacursor_insert_after dcaina
#define dacursor_compare dcacomp
#define dacursor_compare_contents dcacont
#define dacursor_swap dcaswap
#define dacursor_copy daccopy
#define dacursor_move dacmove
#define sdynarray_fgets sdafgets
#define dynarray_dup dnadup
#define dacursor_init_offset dacinitc
#endif

#ifdef _NO_PROTO
sdynarray_t *sdynarray_create();
void sdynarray_destroy();
bool_t sdynarray_write();
bool_t sdynarray_read();
dynarray_t *dynarray_create();
void dynarray_destroy();
void dynarray_reset();
bool_t dynarray_write();
bool_t dynarray_append();
bool_t dynarray_read();
bool_t dacursor_delete();
bool_t dacursor_insert_before();
bool_t dacursor_insert_after();
int dacursor_compare();
int dacursor_compare_contents();
bool_t dacursor_swap();
bool_t dacursor_copy();
bool_t dacursor_move();
dynarray_t *dynarray_dup();
bool_t dacursor_init_offset();
#else
sdynarray_t *sdynarray_create(sdynarray_t *da_ptr);
void sdynarray_destroy(sdynarray_t *da_ptr);
bool_t sdynarray_write(sdynarray_t *da_ptr,void *data,size_t offset,size_t length);
bool_t sdynarray_read(sdynarray_t *da_ptr,void *data,size_t offset,size_t length);
dynarray_t *dynarray_create(dynarray_t *da_ptr,size_t element_size);
void dynarray_destroy(dynarray_t *da_ptr);
bool_t dynarray_write(dynarray_t *da_ptr,void *data,size_t offset);
bool_t dynarray_append(dynarray_t *da_ptr,void *data);
bool_t dynarray_read(dynarray_t *da_ptr,void *data,size_t offset);
bool_t dacursor_delete(dacursor_t *dc_ptr);
bool_t dacursor_insert_before(dacursor_t *dc_ptr,void *data);
bool_t dacursor_insert_after(dacursor_t *dc_ptr,void *data);
int dacursor_compare(dacursor_t *dc1,dacursor_t *dc2,int (*comparison_function)(void *,void *));
int dacursor_compare_contents(dacursor_t *dc1,void *bp,int (*comparison_function)(void *,void *));
bool_t dacursor_swap(dacursor_t *dc1,dacursor_t *dc2);
bool_t dacursor_copy(dacursor_t *to,dacursor_t *from);
bool_t dacursor_move(dacursor_t *from, dacursor_t *to);
dynarray_t *dynarray_dup(dynarray_t *da_ptr, dynarray_t *orig_da_ptr);
bool_t dacursor_init_offset(dacursor_t *dc, dynarray_t *da_ptr, size_t offset);

#endif

#if defined(__cplusplus)
}
#endif

#endif
